# Class for setting up the while / with an overlay
#
# In order to have / directory in overlayfs a special handling at early boot stage is required
# The idea is to supply a custom init script that mounts / (actually /etc) before launching actual init program,
# because the latter already requires /etc to be rw.
#
# The configuration must be machine specific, is coming from the kernel bootparams, set by the bootloader.
#
# The class provides two options for /sbin/init generation
# 1. Default option is to rename original /sbin/init to /sbin/init.orig and place generated init under
#    original name, i.e. /sbin/init. It has an advantage that you won't need to change any kernel
#    parameters in order to make it work, but it poses a restriction that package-management can't
#    be used, becaause updating init manager would remove generated script
# 2. If you are would like to keep original init as is, you can set
#    OVERLAYFS_ROOT_USE_ORIG_INIT_NAME = "0"
#    Then generated init will be named /sbin/preinit and you would need to extend you kernel parameters
#    manually in your bootloader configuration.
#
# Regardless which mode you choose, update and migration strategy of configuration files under /etc
# overlay is out of scope of this class

IMGCLASSES += "overlayfs-root"
IMAGE_FEATURES[validitems] += "overlayfs-root"

ROOTFS_POSTPROCESS_COMMAND += '${@bb.utils.contains("IMAGE_FEATURES", "overlayfs-root", "create_overlayfs_root_preinit;", "", d)}'
IMAGE_FEATURES_CONFLICTS_overlayfs-root = "${@ 'package-management' if bb.utils.to_boolean(d.getVar('OVERLAYFS_ROOT_USE_ORIG_INIT_NAME'), True) else ''}"
IMAGE_FEATURES_CONFLICTS_overlayfs-root += " root-only-rootfs"

OVERLAYFS_ROOT_USE_ORIG_INIT_NAME ??= "1"
OVERLAYFS_ROOT_ORIG_INIT_NAME_SUFFIX ??= ".orig"
OVERLAYFS_ROOT_INIT_TEMPLATE ??= "${@os.path.normpath("${LAYER_META_DIR}/")}/files/overlayfs-root-preinit.sh.in"

python create_overlayfs_root_preinit() {
    import string

    with open(d.getVar("OVERLAYFS_ROOT_INIT_TEMPLATE"), "r") as f:
        PreinitTemplate = string.Template(f.read())

    useOrigInit = oe.types.boolean(d.getVar('OVERLAYFS_ROOT_USE_ORIG_INIT_NAME'))
    useOrigInit = oe.types.boolean(d.getVar('OVERLAYFS_ROOT_USE_ORIG_INIT_NAME'))
    preinitPath = oe.path.join(d.getVar("IMAGE_ROOTFS"), d.getVar("base_sbindir"), "preinit")
    initBaseName = oe.path.join(d.getVar("base_sbindir"), "init")
    origInitNameSuffix = d.getVar('OVERLAYFS_ROOT_ORIG_INIT_NAME_SUFFIX')

    args = {
        'SBIN_INIT_NAME': initBaseName + origInitNameSuffix if useOrigInit else initBaseName
    }

    if useOrigInit:
        # rename original /sbin/init
        origInit = oe.path.join(d.getVar("IMAGE_ROOTFS"), initBaseName)
        bb.debug(1, "rootfs path %s, init path %s, test %s" % (d.getVar('IMAGE_ROOTFS'), origInit, d.getVar("IMAGE_ROOTFS")))
        bb.utils.rename(origInit, origInit + origInitNameSuffix)
        preinitPath = origInit

    with open(preinitPath, 'w') as f:
        f.write(PreinitTemplate.safe_substitute(**args))
    os.chmod(preinitPath, 0o755)
}
